/*
 * FLINT PARTICLE SYSTEM
 * .....................
 * 
 * Author: Richard Lord (Big Room)
 * Copyright (c) Big Room Ventures Ltd. 2008
 * http://flintparticles.org
 * 
 * 
 * Licence Agreement
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.flintparticles.common.initializers 
{
	import org.flintparticles.common.emitters.Emitter;
	import org.flintparticles.common.particles.Particle;
	import org.flintparticles.common.utils.WeightedArray;	

	/**
	 * The ChooseInitializer initializer selects one of multiple initializers, using 
	 * optional weighting values to produce an uneven distribution for the choice, 
	 * and applies it to the particle. This is often used with the InitializerGroup 
	 * initializer to apply a randomly chosen group of initializers to the particle.
	 * 
	 * @see org.flintparticles.common.initializers.InitializerGroup
	 */

	public class ChooseInitializer extends InitializerBase
	{
		private var _initializers:WeightedArray;
		
		/**
		 * The constructor creates a ChooseInitializer initializer for use by 
		 * an emitter. To add a ChooseInitializer to 
		 * an emitter, use the emitter's addInitializer method.
		 * 
		 * @param colors An array containing the Initializers to use.
		 * @param weights The weighting to apply to each initializer. If no weighting
		 * values are passed, the initializers are all assigned a weighting of 1.
		 * 
		 * @see org.flintparticles.common.emitters.Emitter#addInitializer()
		 */
		public function ChooseInitializer( initializers:Array, weights:Array = null )
		{
			_initializers = new WeightedArray;
			var len:int = initializers.length;
			var i:int;
			if( weights != null && weights.length == len )
			{
				for( i = 0; i < len; ++i )
				{
					_initializers.add( initializers[i], weights[i] );
				}
			}
			else
			{
				for( i = 0; i < len; ++i )
				{
					_initializers.add( initializers[i], 1 );
				}
			}
		}
		
		public function addInitializer( initializer:Number, weight:Number = 1 ):void
		{
			_initializers.add( initializer, weight );
		}
		
		public function removeInitializer( initializer:Number ):void
		{
			_initializers.remove( initializer );
		}
		
		/**
		 * @inheritDoc
		 */
		override public function initialize( emitter:Emitter, particle:Particle ):void
		{
			var initializer:Initializer = _initializers.getRandomValue();
			initializer.initialize( emitter, particle );
		}
	}
}
